import { useState, useRef, useCallback } from 'react';
import { dashboardStats, isOnline } from '../services/DashboardService';
import { initiateRides } from '../services/RideService';
import { getFromStorage } from '../utils/storageHelper';
import Geolocation from 'react-native-geolocation-service';
import { PermissionsAndroid, Platform } from 'react-native';
import { useNavigation, useFocusEffect } from '@react-navigation/native';

export const DashboardHook = () => {
  const navigation = useNavigation();
  const [stats, setStats] = useState({ total_earnings: '0', today_earnings: '0', total_rides: 0, rating: 0 });
  const [loading, setLoading] = useState(true);
  const [login_dtl_storage, set_login_dtl_storage] = useState(null);
  const [driver_location, set_driver_location] = useState({ lat: null, lng: null });
  const [initiate_rides, set_initiate_rides] = useState([]);
  const [isEnabled, setIsEnabled] = useState(false);

  const intervalRef = useRef(null); // TIME
  const isPollingStarted = useRef(false); // Prevent duplicate intervals

  useFocusEffect(
    useCallback(() => {
      fetchStorage();
      requestLocationPermission();

      return () => {
        clearPollingInterval();
      };
    }, [])
  );

  const clearPollingInterval = () => {
    if (intervalRef.current) {
      clearInterval(intervalRef.current);
      intervalRef.current = null;
      isPollingStarted.current = false;
    }
  };

  const fetchStorage = async () => {
    const storedData = await getFromStorage('login_dtl');
    const storeParse = JSON.parse(storedData);

    set_login_dtl_storage(storeParse);

    if (storeParse?.id) {
      fetchStats(storeParse.id);
      allinitiateRides(storeParse.latitude, storeParse.longitude);

      if (!isPollingStarted.current) {
        intervalRef.current = setInterval(() => {
          allinitiateRides(storeParse.latitude, storeParse.longitude);
        }, 2000);
        isPollingStarted.current = true;
      }
    }
  };

  const fetchStats = async (driverId) => {
    try {
      const data = await dashboardStats(driverId);
      if (data) {
        setIsEnabled(data?.driver_obj?.availability);
        setStats({
          total_earnings: data.total_earnings ?? '0',
          today_earnings: data.today_earnings ?? '0',
          total_rides: data.total_rides ?? 0,
          rating: data.rating ?? 0,
          currency_icon: data.currency_icon
        });
      }
    } catch (err) {
      console.error('Stats fetch failed', err);
    } finally {
      setLoading(false);
    }
  };

  const requestLocationPermission = async () => {
    if (Platform.OS === 'android') {
      try {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.ACCESS_FINE_LOCATION,
          {
            title: 'Location Permission',
            message: 'This app needs access to your location.',
            buttonNeutral: 'Ask Me Later',
            buttonNegative: 'Cancel',
            buttonPositive: 'OK',
          }
        );
        if (granted === PermissionsAndroid.RESULTS.GRANTED) {
          getLocation();
        } else {
          console.log('Location permission denied');
          setLoading(false);
        }
      } catch (err) {
        console.warn(err);
        setLoading(false);
      }
    } else {
      getLocation(); // iOS
    }
  };

  const getLocation = () => {
    Geolocation.getCurrentPosition(
      (position) => {
        set_driver_location({
          lat: position.coords.latitude,
          lng: position.coords.longitude,
        });
        setLoading(false);
      },
      (error) => {
        console.log('Location error: ', error.message);
        setLoading(false);
      },
      { enableHighAccuracy: true, timeout: 15000, maximumAge: 10000 }
    );
  };

  const allinitiateRides = async (lat, long) => {
    try {
      const response = await initiateRides({ latitude: lat, longitude: long, inprocess: 1 });
      console.log("dashboardhook-initiateRidesResponse===", response);

      if (response) {
        const inProgressRide = response.find(ride =>
          ["ASSIGNED", "ON_THE_WAY", "DRIVER_ARRIVED", "OTP_VERIFY", "START_RIDE", "NEAR_DESTINATION", "PAYMENT_INITIATED", "PAYMENT_COMPLETE"].includes(ride.status_name)
        );

        if (inProgressRide) {
          clearPollingInterval(); // Stop timer

          if (["PAYMENT_INITIATED", "PAYMENT_COMPLETE"].includes(inProgressRide.status_name)) {
            navigation.navigate("Payment", { ride_id: inProgressRide.id });
          } else {
            navigation.navigate("RideStepFirst");
          }
        } else {
          set_initiate_rides(response);
        }
      }
    } catch (error) {
      console.error("Error fetching initiate rides:", error);
    }
  };

  const toggleSwitch = async () => {
    const newStatus = isEnabled == 1 ? 0 : 1;
    setIsEnabled(newStatus);
    try {
      const is_online_resp = await isOnline();
      if (is_online_resp) {
        // setIsEnabled(is_online_resp?.availability)
      }
    } catch (error) {
      // console.error('Error updating status:', error);
    }
  };

  return {
    stats,
    loading,
    login_dtl_storage,
    driver_location,
    initiate_rides,
    isEnabled,
    toggleSwitch
  };
};
