<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;

class FileUploadService
{
    /**
     * Handle the file upload.
     *
     * @param UploadedFile $file
     * @param string $folder
     * @return string
     */
    public function upload(UploadedFile $file, string $folder): string
    {
        $filename = time() . '_' . $file->getClientOriginalName();
        $filePath = $file->storeAs('uploads/' . $folder, $filename, 'public');
        return $filePath;
    }

    /**
     * Handle multiple file uploads.
     *
     * @param array $files
     * @return array
     */
    public function uploadMultiple_oldd(array $files): array
    {
        $filePaths = [];
        foreach ($files as $key => $file) {
            if ($file instanceof UploadedFile) {
                $filePaths[$key] = $this->upload($file, $key);
            }
        }
        return $filePaths;
    }

    public function uploadMultiple(array $files): array
    {
        \Log::info('uploadservice==', [$files]);

        $filePaths = [];
        foreach ($files as $key => $fileArray) {
            $folder = $this->getFolderName($key);
            if ($folder) {

                // Check if the file array is actually an array, if not make it an array
                if (!is_array($fileArray)) {
                    $fileArray = [$fileArray];
                }

                foreach ($fileArray as $file) {
                    if ($file instanceof UploadedFile) {
                        $filename = $this->generateFileName($file, $key);
                        $this->ensureDirectoryExists('uploads/' . $folder);

                        try {
                            $filePath = $file->storeAs('uploads/' . $folder, $filename, 'public');
                            $filePaths[$key] = $filePath;
                        } catch (\Exception $e) {
                            \Log::error('Failed to upload file: ' . $e->getMessage());
                            throw new RuntimeException('File upload failed: ' . $e->getMessage());
                        }
                    }
                }
            }
        }

        return $filePaths;
    }

    

   public function uploadMultiple_old_1(array $files): array
    {
        \Log::info('uploadservice==', [$files]);

        $filePaths = [];
        foreach ($files as $key => $fileArray) {
            $folder = $this->getFolderName($key);
            if ($folder) {
                if (is_array($fileArray)) {
                    \Log::info('innerloop==', $fileArray);
                    foreach ($fileArray as $file) {
                        if ($file instanceof UploadedFile) {
                            $filename = $this->generateFileName($file, $key);
                            $this->ensureDirectoryExists('uploads/' . $folder);

                            try {
                                $filePath = $file->storeAs('uploads/' . $folder, $filename, 'public');
                                $filePaths[$key] = $filePath;
                            } catch (\Exception $e) {
                                \Log::error('Failed to upload file: ' . $e->getMessage());
                                throw new \RuntimeException('File upload failed: ' . $e->getMessage());
                            }
                        }
                    }
                }
            }
        }

        return $filePaths;
    }
 

    public function uploadMultiple_old(array $files): array
    {
        \Log::info('uploadservice==', [
            $files
        ]);

        $filePaths = [];
        foreach ($files as $key => $file) {
            if ($file instanceof UploadedFile) {
                $folder = $this->getFolderName($key);
                $filename = $this->generateFileName($file, $key);

                // Ensure the directory exists
                $this->ensureDirectoryExists('uploads/' . $folder);

                try {
                    $filePath = $file->storeAs('uploads/' . $folder, $filename, 'public');
                    $filePaths[$key] = $filePath;
                } catch (\Exception $e) {
                    // Handle the exception and log the error
                    \Log::error('Failed to upload file: ' . $e->getMessage());
                    throw new \RuntimeException('File upload failed: ' . $e->getMessage());
                }
            }
        }

        return $filePaths;
    }

    /**
     * Ensure the directory exists.
     *
     * @param string $path
     * @return void
     */
    protected function ensureDirectoryExists(string $path): void
    {
        $fullPath = storage_path('app/public/' . $path);
        if (!is_dir($fullPath)) {
            if (!mkdir($fullPath, 0755, true)) {
                throw new \RuntimeException('Failed to create directories: ' . $fullPath);
            }
        }
    }

    /**
     * Generate a file name based on the uploaded file.
     *
     * @param UploadedFile $file
     * @param string $key
     * @return string
     */
    protected function generateFileName(UploadedFile $file, string $key): string
    {
        return $key . '_' . time() . '.' . $file->getClientOriginalExtension();
    }

    /**
     * Get the folder name based on the file key.
     *
     * @param string $key
     * @return string
     */
    protected function getFolderName(string $key): string
    {
        $folders = [
            'registration_certificate'  => 'rc_files',
            'vehicle_insurance'         => 'vi_files',
            'driving_licence'           => 'vl_files',
            'profile_photo'             => 'profile_photos',
            'vehicle_photo'             => 'vehicle_photo'
        ];

        return $folders[$key] ?? 'other_files';
    }

}

