<?php
namespace App\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

use Illuminate\Support\Facades\Http;

use App\Models\Parcels;
use App\Models\Category;
use App\Models\ParcelRequest;

use App\Repositories\ParcelRepository;
use App\Repositories\DriverRepository;

use App\Services\UploadDocService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class ParcelService
{
    protected $parcelRepository;
    protected $driverRepository;
    protected $driverService;

    public function __construct(ParcelRepository $parcelRepository, DriverRepository $driverRepository, UploadDocService $driverService)
    {
        $this->parcelRepository = $parcelRepository;
        $this->driverRepository = $driverRepository;
        $this->driverService    = $driverService;
    }

///////////USER//////////////
    public function createParcel(array $data)
    {
        return $this->parcelRepository->createParcel($data);
    }

    public function getParcel($id)
    {
        $parcel = $this->parcelRepository->findById($id);
        if(!$parcel)
        return false;

        $distance = $this->calculateDistance($parcel->delivery_latitude, $parcel->delivery_longitude, $parcel->pickup_latitude, $parcel->pickup_longitude);
        $parcel->distance   = $distance;
        $parcel->price      = $this->calculateEarnings($parcel, '');;
        return $parcel;
    }

    public function isRideAccept()
    {
        $parcel = $this->parcelRepository->isAcceptRide();
        return $parcel;
    }

    public function inProcessRide($userId)
    {
       $parcel = $this->parcelRepository->inProcessRide($userId);

        if($parcel) {
            $parcel->screen_name = ScreenNameForUser($parcel->status);
            $parcel->status_name = statusName($parcel->status);
            $parcel->currency = '$';
            $parcel->distance_unit = 'KM';
            return [$parcel];
        }
        return false;
 
    }

    public function startRideStatus($userId, $parcelId)
    {
        $parcel = $this->parcelRepository->startRideStatus($userId, $parcelId);
        if($parcel) {
            return $parcel;
        }     
        return false;
    }
    
    public function rideTrack($userId)
    {
        $parcel = $this->parcelRepository->rideTrack($userId);
        if($parcel) {
            return $parcel;
        }     
        return false;
    } 

    public function allRides($user_id)
    {
        $trips = $this->parcelRepository->allRides($user_id);
        return $trips;
    }

    public function allCoupon()
    {
        $couponArr = $this->parcelRepository->allCoupon();
        return $couponArr;
    }

    public function cancelParcelByUser(int $parcelId, int $userId, string $reason): bool
    {
        return $this->parcelRepository->cancelParcelByUser($parcelId, $userId, $reason);
    } 

/////////////////////////////

    public function getLatestParcel()
    {
        $parcel = $this->parcelRepository->getLatestParcel();

        if($parcel) {
            $parcel->currency = '$';
            return [$parcel];
        }
        return false;
    }

    public function trackParcel($parcelId, $driverId, $status)
    {
        $parcel = $this->parcelRepository->acceptAfterParcel($parcelId, $driverId, $status);

        if($parcel) {
            $parcel->currency = '$';
            return [$parcel];
        }
        return false;
    }

    public function acceptAfterParcel($parcelId, $driverId)
    {
        $parcel = $this->parcelRepository->acceptAfterParcel($parcelId, $driverId);

        if($parcel) {
            $parcel->currency = '$';
            return [$parcel];
        }
        return false;
    }

    public function getPendingParcelsForDriver($driver_id)
    {
        return $this->parcelRepository->getPendingParcels();
    }

    public function getNearbyParcels($driver, $status, $latitude, $longitude, $radius)
    {
        $parcels = $this->parcelRepository->getNearbyParcels($status, $latitude, $longitude, $radius);

        $parcels = $parcels->map(function($parcel) use ($latitude, $longitude, $driver) {
            $distance = $this->calculateDistance($latitude, $longitude, $parcel->pickup_latitude, $parcel->pickup_longitude);
            $duration = $this->calculateDuration($distance);
            $earning = $this->calculateEarnings($parcel, $driver);

            return [
                'parcel_id'=> $parcel->id,
                'distance' => sprintf('%.2f', $distance).' Km',
                'duration' => sprintf('%.2f', $duration). ' Minute',
                'earning' =>  '₹'.sprintf('%.2f', $earning),
            ];
        });
        return $parcels;

        /*return $parcels->map(function ($parcel) use ($latitude, $longitude, $driver) {
            $parcel->distance = $this->calculateDistance($latitude, $longitude, $parcel->pickup_latitude, $parcel->pickup_longitude);
            $parcel->duration = $this->calculateDuration($parcel->distance);
            $parcel->earning = $this->calculateEarnings($parcel, $driver);
            return $parcel;
        });*/
    }

    private function calculateDistance($lat1, $lon1, $lat2, $lon2)
    {
        $earthRadius = 6371;
        $latDistance = deg2rad($lat2 - $lat1);
        $lonDistance = deg2rad($lon2 - $lon1);

        $a = sin($latDistance / 2) * sin($latDistance / 2) +
            cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
            sin($lonDistance / 2) * sin($lonDistance / 2);

        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
        return sprintf('%.2f', ($earthRadius * $c));
    }

    private function calculateDuration($distance)
    {
        $speed = 40;
        $duration = ($distance / $speed) * 60; // Duration in minutes
        return sprintf('%.2f', $duration);
    }

    private function calculateEarnings($parcel, $driver)
    {
        $baseFare = 50; // Base fare
        $perKmRate = 10; // Rate per kilometer
        $distanceEarning = $parcel->distance * $perKmRate;

        $earnings = $baseFare + $distanceEarning;
        return sprintf('%.2f', $earnings);
    }

    public function parcelRequestStatus($parcelId, $driverId, $status)
    {
       ParcelRequest::updateOrCreate(
                ['parcel_id' => $parcelId, 'driver_id' => $driverId],
                ['status' => $status]
            ); 
    }

    public function rejectParcel($parcelId, $driverId)
    {
        return DB::transaction(function () use ($parcelId, $driverId) {
            $parcel = Parcels::find($parcelId);

            if (!$parcel || $parcel->status !== '1') {
                return false;
            }

            // Insert into parcel_requests with status 'rejected'
            ParcelRequest::updateOrCreate(
                ['parcel_id' => $parcelId, 'driver_id' => $driverId],
                ['status' => CANCELED]
            );
        

            // Make the parcel available for other drivers
            $parcel->driver_id = null;
            $parcel->status             = '1'; // Ensure parcel is visible to others
            return $parcel->save();
        });
    }

    public function activeRide($driverId)
    {
        return Parcels::where('driver_id', $driverId)
            ->whereIn('status', [
                ASSIGNED,
                ON_THE_WAY,
                DRIVER_ARRIVED,
                OTP_VERIFY,
                START_RIDE,
                NEAR_DESTINATION,
                PAYMENT_INITIATED,
                PAYMENT_COMPLETE
            ])
            ->first();
        

    }

    public function acceptParcel($parcelId, $driverId, $isScheduleAccept, $isQueued)
    {
        \Log::info("acceptride: ", [$parcelId, $driverId, $isScheduleAccept, $isQueued]);
        return DB::transaction(function () use ($parcelId, $driverId, $isScheduleAccept, $isQueued) {
            $parcel = Parcels::find($parcelId);

            if (!$parcel || !in_array($parcel->status , ['1','11'])) {
                return false;
            }
            $is_active_ride = $this->activeRide($driverId);
            $assigned = $is_active_ride ? QUEUED : ASSIGNED;

            // Insert into parcel_requests with status 'accepted'
            ParcelRequest::updateOrCreate(
                ['parcel_id' => $parcelId, 'driver_id' => $driverId],
                ['status' => $assigned]
            );

            // Assign the parcel to the driver
            $otp = Str::random(6); 
            $otp = 1234;

            $parcel_status = $assigned;
            #if($parcel->schedule_ride){
            if($parcel->schedule_ride && !$isScheduleAccept){
            $parcel_status =  SCHEDULE_RIDE;
            }else if($isQueued){
            $parcel_status = QUEUED;
            }


            $parcel->driver_id      = $driverId;
            $parcel->pickup_otp     = $otp;
            $parcel->delivery_otp   = $otp;
            $parcel->status         = $parcel_status;

            $parcel->save();
            return $parcel;
        });
    }

    public function onTheWay($parcelId, $driverId, $status)
    {
        return DB::transaction(function () use ($parcelId, $driverId, $status) {
            $parcel = Parcels::find($parcelId);
            if($status == ON_THE_WAY && $parcel->status != ASSIGNED){
                return false;
            }
            $parcel->status             = ON_THE_WAY;
            return $parcel->save();
        });
    }

    public function ridePayment($parcelId, $driverId, $status)
    {
        return DB::transaction(function () use ($parcelId, $driverId, $status) {
            $parcel = Parcels::find($parcelId);
            if($status == PAYMENT_INITIATED && $parcel->status != START_RIDE){
                return false;
            }
            $parcel->status             = PAYMENT_INITIATED;
            $parcel->save();
            return $parcel->fresh(['driver', 'user']);
        });
    }

    public function startParcelNew($parcelId, $driverId, $status)
    {
        return DB::transaction(function () use ($parcelId, $driverId, $status) {
            $parcel = Parcels::find($parcelId);
            if (!$parcel)
            return ['success' => false, 'message' => 'Parcel not found.'];

            if($status == DRIVER_ARRIVED && $parcel->status != ON_THE_WAY){
                return ['success' => false, 'message' => 'Driver can only arrive when ride is on the way.'];
            }elseif($status == START_RIDE && $parcel->status != OTP_VERIFY){
                return ['success' => false, 'message' => 'Cannot start ride unless OTP verified.'];
            }
            
            /*if (!$parcel || ($status != 7 && $parcel->status !== '2')) 
            {
                return false;
            }*/

            $parcel->status             = "$status"; // Mark parcel as start on track

            if($status == 9)
            {
                $pickupLatLong = [];
                $pickupLatLong['pickup_latitude']   = $parcel->pickup_latitude;
                $pickupLatLong['pickup_longitude']  = $parcel->pickup_longitude;
                
                $this->driverRepository->updateLatLongAsPicup($driverId, $pickupLatLong); 
            }
            $parcel->save();
            return [
            'success' => true,
            'message' => 'Parcel status updated successfully.',
            'parcel' => $parcel->fresh(['user']),
            ];
        });
    }

    public function startParcel($parcelId, $driverId, $status)
    {
        return DB::transaction(function () use ($parcelId, $driverId, $status) {
            $parcel = Parcels::find($parcelId);
            if (!$parcel)
            return ['success' => false, 'message' => 'Parcel not found.'];

            if($status == DRIVER_ARRIVED && $parcel->status != ON_THE_WAY){
                return ['success' => false, 'message' => 'Driver can only arrive when ride is on the way.'];
            }elseif($status == START_RIDE && $parcel->status != OTP_VERIFY){
                return ['success' => false, 'message' => 'Cannot start ride unless otp verified.'];
            }

            $parcel->status             = "$status";
            if($status == START_RIDE)
            $parcel->pickup_time = now();

            if($status == DRIVER_ARRIVED)
            {
                $pickupLatLong = [];
                $pickupLatLong['pickup_latitude']   = $parcel->pickup_latitude;
                $pickupLatLong['pickup_longitude']  = $parcel->pickup_longitude;
                
                $this->driverRepository->updateLatLongAsPicup($driverId, $pickupLatLong); 
            }
            $parcel->save();
            return [
            'success' => true,
            'message' => 'Parcel status updated successfully.',
            'parcel' => $parcel->fresh(['user']),
            ];
        });
    }

    public function inProcessParcel($driverId)
    {
       $parcel = $this->parcelRepository->inProcessParcel($driverId);

        if($parcel) {
            $parcel->screen_name = getScreenName($parcel->status);
            $parcel->currency = '$';
            return [$parcel];
        }
        return false;
 
    }

    public function cancelParcel(int $parcelId, int $driverId, string $reason): bool
    {
        return $this->parcelRepository->cancelParcel($parcelId, $driverId, $reason);
    }

    public function getParcelSummary(int $parcelId)
    {
        $parcel = $this->parcelRepository->findById($parcelId);
        if (!$parcel) {
            return null;
        }

        $start = Carbon::parse($parcel->updated_at);

        $ret_data = [];
        $ret_data['pickup_name'] = $parcel->contact_name ? $parcel->contact_name : $parcel->user->name;
        $ret_data['user_photo'] = optional($parcel->user)->photo_url ?? asset('images/user.jpg');
        $ret_data['time'] = Carbon::parse($start)->format('d M Y, h:i A');

        return $ret_data;

        /*
        return [
            'pickup_details' => [
                'name' => $parcel->pickup_name,
                'mobile' => $parcel->pickup_mobile,
                'address' => $parcel->pickup_address,
                'latitude' => $parcel->pickup_latitude,
                'longitude' => $parcel->pickup_longitude,
                'time_window_start' => $parcel->pickup_time_window_start,
                'time_window_end' => $parcel->pickup_time_window_end,
            ],
            'delivery_details' => [
                'name' => $parcel->delivery_name,
                'mobile' => $parcel->delivery_mobile,
                'address' => $parcel->delivery_address,
                'latitude' => $parcel->delivery_latitude,
                'longitude' => $parcel->delivery_longitude,
                'time_window_start' => $parcel->delivery_time_window_start,
                'time_window_end' => $parcel->delivery_time_window_end,
            ],
            'status' => $parcel->status,
            'assigned_driver' => $parcel->driver ? $parcel->driver->name : null,
            'assigned_time' => $parcel->assigned_time,
            'description'   => $parcel->description
        ]; */
    
    }     

    public function pickupParcel($driverId, $parcelId, $otp)
    {
            $parcel = Parcels::find($parcelId);

            if(!$parcel || $parcel->status !== '2')
            return 'First Assigned';
            
            if($parcel->pickup_otp !== $otp)
            return 'Invalid OTP';
            
            
            $parcel->status     = '3'; // Mark parcel as pickup
            $parcel->pickup_time= now();
            $parcel->save();

            return $parcel; 
    }

    public function deliverParcel($driverId, $parcelId, $otp)
    {
        return DB::transaction(function () use ($parcelId, $driverId, $otp) {
            $parcel = Parcels::find($parcelId);

            if(!$parcel || !in_array($parcel->status, ['3', '7']) || $parcel->driver_id !== $driverId)
            return 'Invalid Parcel or Driver';
            
            if($parcel->delivery_otp !== $otp)
            return 'Invalid OTP';
            
            
            $parcel->status         = '4'; // Mark parcel as delivered
            $parcel->delivery_time  = now();
            $parcel->save();

            $distanceCovered = $this->calculateDistance($parcel->pickup_latitude, $parcel->pickup_longitude, $parcel->delivery_latitude, $parcel->delivery_longitude);

            $parcel->earning = $this->driverService->calculateAndStoreEarnings($driverId, $parcelId, $distanceCovered);

            return $parcel;
        }); 
    }

    public function historyParcel($driverId)
    {
        return $this->parcelRepository->historyParcel($driverId);
    } 

    public function updatePaymentStatus($driverId, $parcelId)
    {
        return $this->parcelRepository->updatePaymentStatus($driverId, $parcelId);
    }

    public function getVehicleOptions(array $data)
    {
        $pickupLat  = $data['pickup_latitude'];
        $pickupLng  = $data['pickup_longitude'];
        $dropLat    = $data['delivery_latitude'];
        $dropLng    = $data['delivery_longitude'];

        /*
        // Calculate distance using Haversine formula
        $earthRadius = 6371; // in km
        $latDiff = deg2rad($dropLat - $pickupLat);
        $lngDiff = deg2rad($dropLng - $pickupLng);

        $a = sin($latDiff/2) * sin($latDiff/2) +
            cos(deg2rad($pickupLat)) * cos(deg2rad($dropLat)) *
            sin($lngDiff/2) * sin($lngDiff/2);
        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
        $distance = $earthRadius * $c;

        // Approximate time (in minutes)
        $averageSpeed = 30; // km/h
        $time = ceil(($distance / $averageSpeed) * 60);*/

        $distance_time = $this->getDistanceAndDuration($pickupLat, $pickupLng, $dropLat, $dropLng); 
        $distance= 0 ; $time=0;
        if($distance_time['distance_meters']){
        $distance_km = ($distance_time['distance_meters'] / 1000);
        $distance = number_format($distance_km, 2, '.', '');
        $time = round($distance_time['duration_seconds'] / 60);
        }
   
        $query = Category::where('status', '1');
        if(isset($data['call_type']) && $data['call_type'] == 'inner') {
            $vehicle_id = isset($data['vehicle_id']) && $data['vehicle_id'] ? $data['vehicle_id'] : 1;
            $query->where('id', $vehicle_id);
        }
        
        $vehicles = $query->get();

        $options = collect($vehicles)->map(function ($vehicle) use ($distance, $time) {
            $price = $vehicle['base_fare'] + ($vehicle['fare_per_km_for_extra_run'] * $distance);
            return [
                'type' => $vehicle['name'],
                'estimated_time' => $time .' mins',
                'estimated_time_mints' => $time,
                'estimated_price' => round($price),
                'distance'=> $distance,
                'currency'=> currency_icon()
            ];
        });

        return $options; 
    }


function getDistanceAndDuration($originLat, $originLng, $destLat, $destLng)
{
    $apiKey = GOOGLE_MAPS_API_KEY;
    $url = "https://maps.googleapis.com/maps/api/distancematrix/json";

    $response = Http::get($url, [
        'origins' => "$originLat,$originLng",
        'destinations' => "$destLat,$destLng",
        'key' => $apiKey,
        'mode' => 'driving', // or walking, bicycling, transit
        'language' => 'en'
    ]);

    $data = $response->json();

    if ($data['status'] === 'OK' && $data['rows'][0]['elements'][0]['status'] === 'OK') {
        $element = $data['rows'][0]['elements'][0];
        return [
            'distance' => $element['distance']['text'],         // e.g. "5.9 km"
            'duration' => $element['duration']['text'],         // e.g. "12 mins"
            'distance_meters' => $element['distance']['value'], // in meters
            'duration_seconds' => $element['duration']['value'] // in seconds
        ];
    }

    return ['distance'=>0, 'duration'=>0, 'distance_meters'=>0, 'duration_seconds'=>0];
}

    public function queuedToAssign($driverId)
    {
        return $this->parcelRepository->queuedToAssign($driverId);
    }

    
    

}
