<?php
namespace App\Http\Controllers\Api\Driver;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

use App\Services\ParcelService;
use App\Http\Requests\ParcelRequst;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

use App\Models\Parcels;
use App\Models\Drivers;
use App\Models\Feedback;
use Carbon\Carbon;


class ParcelController extends Controller
{
	protected $parcelService;

	public function __construct(ParcelService $parcelService)
	{
		$this->parcelService = $parcelService;
	}

    public function singleMode(ParcelRequst $request)
    {
        $data = $request->validated();
        $data['driver_id'] = $request->login_detail['login_id'];
        $parcel = $this->parcelService->createParcel($data);

        return apiResponse(1, 'Single Mode ride created successfully.', $parcel, 200);
    }		

    public function checkNewParcel(Request $request)
    {
        $driver_id = $request->login_detail['login_id'];
        $driver_obj = Drivers::find($driver_id);
        if(!$driver_obj->availability)
        return apiResponse(1, 'You are OffLine', '', 200);
        
        $latestParcel = $this->parcelService->getLatestParcel();
        if($latestParcel)
        return apiResponse(1, 'Latest Parcel', $latestParcel, 200);
        else
        return apiResponse(0, 'There is no new parcel', [], 200);
    }	

	public function getPendingParcels(Request $request)
	{
		$login_detail 	= $request->login_detail;	
		$driver_login_id= $login_detail['login_id'];

		$parcels = $this->parcelService->getPendingParcelsForDriver($driver_login_id);
        
		return apiResponse(1, 'Pending Parcels', $parcels, 200);	
	}

    public function getNearbyParcelsOld(ParcelRequst $request)
    {
        $login_detail   = $request->login_detail;   
        $driver_login_id= $login_detail['login_id'];

        $pendingRides = Parcels::with(['user:id,name,mobile,photo'])
            ->whereIn('status', [PENDING, ASSIGNED])
            ->whereNotIn('id', function ($query) use ($driver_login_id) {
                $query->select('parcel_id')
                ->from('parcel_requests')
                ->where('status', '2')
                ->where('driver_id', $driver_login_id);
            })
            ->orderby('id', 'DESC')
            ->get(['id', 'user_id', 'pickup_address', 'delivery_address', 'final_price', 'status']);
        
        return apiResponse(1, '', $pendingRides, 200);

        //comment for the timing

        $validated = $request->validated();
        $parcels = $this->parcelService->getNearbyParcels(
            $login_detail,
            $validated['status'] ?? '1',
            $validated['latitude'],
            $validated['longitude'],
            $validated['radius'] ?? 10
        );

        return apiResponse(1, 'Near By Parcels', $parcels, 200);
    }

    public function getNearbyParcels(ParcelRequst $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];

        
        $driver = Drivers::find($driver_login_id);
            if (!$driver || $driver->availability != 1) {
                return apiResponse(1, 'Driver not available for new rides', [], 200);
            } 

        $validated = $request->validated();


        $fields = [
            'id', 'user_id', 'driver_id', 'pickup_address', 'pickup_latitude', 'pickup_longitude',
            'delivery_address', 'delivery_latitude', 'delivery_longitude',
            'distance', 'final_price', 'status', 'schedule_ride'
        ];

        $statuses = [ASSIGNED, ON_THE_WAY, DRIVER_ARRIVED, OTP_VERIFY, START_RIDE, NEAR_DESTINATION, PAYMENT_INITIATED, PAYMENT_COMPLETE, PENDING, COMPLETE];
        if(isset($request->inprocess) && $request->inprocess == 1)
        $statuses = array_values(array_filter($statuses, fn($status) => $status !== COMPLETE));

        $pendingRides = collect($statuses)->flatMap(function ($status) use ($driver_login_id, $fields) {
                $query = Parcels::with(['user:id,name,mobile,photo', 'driver:id,latitude,longitude,availability'])
                ->select($fields)
                ->where('status', $status)
                ->whereNotIn('id', function ($q) use ($driver_login_id) {
                        $q->select('parcel_id')
                        ->from('parcel_requests')
                        ->where('status', CANCELED)
                        ->where('driver_id', $driver_login_id);
                        });

                if ($status === COMPLETE) {
                    $query->where('driver_id', $driver_login_id);
                    $parcels = $query->orderByDesc('id')->get();
                } elseif ($status !== PENDING) {
                    $query->where('driver_id', $driver_login_id);
                     $parcels = collect([$query->first()])->filter();
                } else {
                  $parcels = $query->orderByDesc('id')->get();
                }

                return $parcels->map(function ($parcel) use ($status) {
                        $parcel->estimated_time = $status === COMPLETE ? null : '15 mins';
                        $parcel->distance_unit = 'km';
                        return $parcel;
                        });
                });

        $hasActive = $pendingRides->contains(fn($ride) =>
            in_array($ride->status, [NEAR_DESTINATION, PAYMENT_INITIATED, PAYMENT_COMPLETE])
            );


        $hasAccepted = DB::table('parcel_requests')
            ->where('driver_id', $driver_login_id)
            ->where('status', 1)
            ->exists();

        $queuedRides = collect();
        if($hasActive && !$hasAccepted) {
            $queuedRides = Parcels::with(['user:id,name,mobile,photo', 'driver:id,latitude,longitude'])
                ->select($fields)
                ->where('status', PENDING)
                ->whereNotIn('id', function ($q) use ($driver_login_id) {
                        $q->select('parcel_id')->from('parcel_requests')->where('status', CANCELED)->where('driver_id', $driver_login_id);
                        })->orderByDesc('id')->get()
                ->map(function ($p) { 
                                $p->estimated_time = '14 mins'; 
                                $p->distance_unit = 'km'; 
                                return $p; 
                                });

        }

        $allRides = $pendingRides->merge($queuedRides)->unique('id')->values();
        return apiResponse(1, '', $allRides, 200);

    }
 
    

    public function rejectParcel(Request $request)
    {
        $login_detail   = $request->login_detail;   
        $driverId = $login_detail['login_id'];


        if ($this->parcelService->rejectParcel($request->parcel_id, $driverId)) {
            return apiResponse(1, 'Parcel rejected successfully.', '', 200);
        }

        return apiResponse(0, 'Failed to reject parcel.', '', 400);
    }

    public function acceptParcel(ParcelRequst $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];
        $is_schedule    = $request->is_schedule;
        $is_queued      = $request->is_queued;



        $parcel_info = $this->parcelService->acceptParcel($request->parcel_id, $driver_login_id, $is_schedule, $is_queued);
        if ($parcel_info) {
            return apiResponse(1, 'Parcel accepted successfully.', $parcel_info, 200);
        }

        return apiResponse('1', 'Failed to accept parcel.', '', 400);
    }

    public function acceptAfterParcel(Request $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];


        $data = $this->parcelService->acceptAfterParcel($request->parcel_id, $driver_login_id);
        if ($data) {
            return apiResponse(1, 'Start Ride.', $data, 200);
        }

        return apiResponse('1', 'Failed to accept parcel.', '', 400);
    }

    public function inProcessParcel(Request $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];

        #$driver = Drivers::where('id', $driver_login_id)->first();

        $data = $this->parcelService->inProcessParcel($driver_login_id);
        if ($data) {
            return apiResponse(1, 'In Process Ride.', $data, 200);
        }

        return apiResponse('1', 'Failed to get parcel.', '', 400);
    }

    public function ArrivedPickupLocation(Request $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];


        $data = $this->parcelService->startParcel($request->parcel_id, $driver_login_id, DRIVER_ARRIVED);
        if($data['success']) {
            return apiResponse(1, $data['message'], $data['parcel'], 200);
        }

        return apiResponse('0', $data['message'], 'null', 400);
    }

    public function startParcel(Request $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];


        $data = $this->parcelService->startParcel($request->parcel_id, $driver_login_id, START_RIDE);
        if($data['success']) {
            return apiResponse(1, $data['message'], $data['parcel'], 200);
        }

        return apiResponse('0', $data['message'], 'null', 400);

    }

    public function trackParcel(Request $request)
    {
        $login_detail   = $request->login_detail;
        $driver_login_id= $login_detail['login_id'];

        $trackParcel = $this->parcelService->trackParcel($request->parcel_id, $driver_login_id, [7]);
        if($trackParcel)
        return apiResponse(1, 'Track Parcel', $trackParcel, 200);
        else
        return apiResponse(0, 'There is no ride', [], 200);
    }

    public function cancelParcel(ParcelRequst $request)
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();
        $parcelId   = $validated['parcel_id'];
        $reason     = $validated['reason'];
        $driverId   = $login_detail['login_id'];

        $success = $this->parcelService->cancelParcel($parcelId, $driverId, $reason);

        if($success)
        return apiResponse(1, 'Parcel canceled successfully.', $success, 200);

        return apiResponse(0, 'Parcel could not be canceled.', '', 400);

    }

    public function parcelSummary(ParcelRequst $request)
    {
        $validated  = $request->validated();
        $parcelId   = $validated['parcel_id'];

        $parcelSummary = $this->parcelService->getParcelSummary($parcelId);

        if(!$parcelSummary)
        return apiResponse(0, 'Parcel not found.', '', 404);

        return apiResponse(1, '', $parcelSummary, 200);
    }

    public function pickupParcel(ParcelRequst $request) 
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();
        $parcelId   = $validated['parcel_id'];
        $otp        = $validated['otp'];  

        $parcel = $this->parcelService->pickupParcel($login_detail['login_id'], $parcelId, $otp); 

        if(is_string($parcel))
        return apiResponse(0, $parcel, '', 400);   
        
        return apiResponse(1, 'Parcel successfully picked up', $parcel, 200);
    }

    public function deliverParcel(ParcelRequst $request) 
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();
        $parcelId   = $validated['parcel_id'];
        $otp        = '1234'; //$validated['otp'];  

        $parcel = $this->parcelService->deliverParcel($login_detail['login_id'], $parcelId, $otp); 

        if(is_string($parcel))
        return apiResponse(0, $parcel, '', 400);   
        
        return apiResponse(1, 'Parcel successfully delivered', $parcel, 200);
    }

    public function historyParcel(Request $request)
    {
        $login_detail= $request->login_detail;
    
        $parcel = $this->parcelService->historyParcel($login_detail['login_id']);

        if($parcel)
        return apiResponse(1, 'Parcel Delivered List', $parcel, 200);
        
        return apiResponse(0, 'There is no data', '', 200); 
    }

    public function paymentReceived(ParcelRequst $request)
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();
        
        $parcel = $this->parcelService->updatePaymentStatus($login_detail['login_id'], $validated['parcel_id']);

        return apiResponse(1, 'Payment Received successfully.', $parcel, 200);
        
    }

    public function dashboardStats(Request $request)
    {
        $driverId = $request->login_detail['login_id'];
        $today = Carbon::today()->toDateString();

        $data = Parcels::selectRaw("
            COALESCE(SUM(final_price), 0) as total_earnings,
            COALESCE(SUM(CASE WHEN DATE(created_at) = '$today' THEN final_price ELSE 0 END), 0) as today_earnings,
            COUNT(*) as total_rides
        ")
        ->where('driver_id', $driverId)
        ->where('status', COMPLETE)
        ->first();

        $data->rating = 4;
        $data->driver_obj = Drivers::find($driverId);
        return apiResponse(1, '', $data, 200);
         
    }

    public function driverOnTheWay(ParcelRequst $request)
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();

        $parcel = $this->parcelService->onTheWay($validated['parcel_id'], $login_detail['login_id'], ON_THE_WAY);
        if($parcel)
        return apiResponse(1, 'On The Way.', $parcel, 200); 
        else
        return apiResponse(0, 'Something went wrong.', [], 200);
       
    } 
 
public function otpVerify(Request $request)
{
    $login_detail   = $request->login_detail;
    $driver_login_id= $login_detail['login_id'];

    if($request->parcel_id){
        $parcelId = $request->parcel_id;
        $parcel = Parcels::with(['user:id,name,mobile,photo'])
        ->find($parcelId);
        if($parcel->delivery_otp === $request->otp)
        {
            $parcel->status = OTP_VERIFY;
            $parcel->save();
            return apiResponse(1, 'OTP Verify.', $parcel->fresh(['user:id,name,mobile,photo']), 200);
        }else{
            return apiResponse(0, 'Not Valid.', [], 200);
        }
    } else {
        return apiResponse(0, 'Ride Id required.', [], 200);
    }
}  

    public function ridePaymentInitiated(ParcelRequst $request)
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();

        $parcel = $this->parcelService->ridePayment($validated['parcel_id'], $login_detail['login_id'], PAYMENT_INITIATED);
        if($parcel)
        return apiResponse(1, 'Payment Initiated.', $parcel, 200); 
        else
        return apiResponse(0, 'Something went wrong.', [], 200);
       
    }

    public function ridePaymentInitiateFare(ParcelRequst $request)
    {
        $login_detail= $request->login_detail;
        $driver_login_id= $login_detail['login_id'];

        $validated  = $request->validated();

        //\Log::info("paymentinitiate: ", [$driver_login_id, $validated['parcel_id']]);

        $rideObj = Parcels::where('driver_id', $driver_login_id)
                        ->where('id', $validated['parcel_id'])
                    ->first();

        if(!$rideObj)
        apiResponse(1, 'There is no ride', [], 200);

        $data = [];
        $data['call_type'] = 'inner';
        $data['pickup_latitude']    = $rideObj->pickup_latitude;
        $data['pickup_longitude']   = $rideObj->pickup_longitude;
        $data['delivery_latitude']  = $rideObj->delivery_latitude;
        $data['delivery_longitude'] = $rideObj->delivery_longitude;
        $data['vehicle_id']         = $rideObj->vehicle_id;   
 
        $vehicle_info = $this->parcelService->getVehicleOptions($data);

        $ride_summary = [];
        $ride_summary['ride_id']    = $rideObj->id; 
        $ride_summary['final_price']= $vehicle_info[0]['estimated_price']; 
        $ride_summary['from']       = $rideObj->pickup_address; 
        $ride_summary['drop']       = $rideObj->delivery_address; 
        $ride_summary['distance']   = $vehicle_info[0]['distance']; 
        $ride_summary['distance_unit']   = 'Km'; 

        $start = Carbon::parse($rideObj->pickup_time);
        $end   = Carbon::parse($rideObj->delivery_time);

        $diff = $start->diff($end);

        $ride_summary['start_time'] = Carbon::parse($start)->format('d M Y, h:i A');
        $ride_summary['end_time']   = Carbon::parse($end)->format('d M Y, h:i A');
        $ride_summary['total_time'] = trim(
            ($diff->h ? $diff->h . ' hour' . ($diff->h > 1 ? 's ' : ' ') : '') .
            ($diff->i ? $diff->i . ' minute' . ($diff->i > 1 ? 's' : '') : '')
        );
        $ride_summary['currency_icon'] = $rideObj->currency_icon;

        return apiResponse(1, '', $ride_summary, 200);
    } 


    public function feedbackForUser(ParcelRequst $request)
    {
        $login_detail= $request->login_detail;
        $driver_login_id= $login_detail['login_id'];

        $validated  = $request->validated();
       

 
        $feedback = Feedback::create([
        'created_by' => $driver_login_id,
        'from_provider' => 'driver',
        'rating' => $validated['rating'],
        'message' => $validated['message'] ?? null,
    ]);

    return apiResponse(1, 'Feedback submit successfully.', $feedback, 200);
     
             
    }

    public function cancelReason()
    {
        return apiResponse(1, '', reasonList(), 200);
    } 

}

