<?php

namespace App\Http\Controllers\Api\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Http\Requests\ParcelRequst;
use App\Services\ParcelService;
use Illuminate\Support\Facades\Storage;

use App\Models\Parcels;
use App\Models\Discount;
use Carbon\Carbon;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ParcelController extends Controller
{

    protected $parcelService;

    public function __construct(ParcelService $parcelService)
    {
        $this->parcelService    = $parcelService;
    }    

    public function store(ParcelRequst $request)
    {
        $userId = $request->user_login['login_id'];
        $in_process_ride = $this->inProcessRide($request, 'inner');
        if($in_process_ride){
            return apiResponse(1, 'You have already created ride.', $in_process_ride, 200);           
        }
        
        
        $data = $request->validated() + [
            'user_id'          => $request->user_login['login_id'],
            'payment'          => '0.00',
            'payment_type'     => '1',
            'payment_received' => '0',
            'commission'       => '0.00',
            'coupon_id'        => null,
            'discount_price'   => null,
        ];

        #\Log::info('creatride: ', $request->vehicle_data);
        if (!empty($request->vehicle_data['id'])) {
            $vehicle = $request->vehicle_data;
        } else {
            $data['call_type'] = 'inner';
            $vehicle_info = $this->parcelService->getVehicleOptions($data);
            $vehicle = $vehicle_info[0] ?? null;
        }

        if ($vehicle) {
            $data['vehicle_id']     = $vehicle['id'];
            $data['distance']       = $vehicle['distance'];
            $data['travel_time']    = $vehicle['estimated_time_mints'];
            $data['payment']        = $vehicle['estimated_price'];
            $data['final_price']    = $vehicle['estimated_price'];
        }

        if (!empty($request->coupon)) {
            $data['coupon_id']      = $request->coupon['coupon_id'] ?? null;
            $data['discount_price'] = $request->coupon['discount_value'] ?? null;
            $data['payment']        = $request->coupon['estimated_price'] ?? $data['payment'];
            $data['final_price']    = $request->coupon['final_price'] ?? $data['final_price'];
        }

        if (!empty($data['schedule_ride'])) {
            $data['schedule_ride'] = Carbon::createFromFormat('d-M Y H:i', $data['schedule_ride'])->format('Y-m-d H:i:s');
        }

        #\Log::info('rideCreate:', $data);
            
        $parcel = $this->parcelService->createParcel($data);

        return apiResponse(1, 'Parcel created successfully.', $parcel, 200);
    }
 

    public function cancelParcel(ParcelRequst $request)
    {
        $login_detail= $request->login_detail;

        $validated  = $request->validated();
        $parcelId   = $validated['ride_id'];
        $reason     = $validated['reason'];
        $userId     = $request->user_login['login_id'];

        $success = $this->parcelService->cancelParcelByUser($parcelId, $userId, $reason);

        if($success)
        return apiResponse(1, 'Parcel canceled successfully.', $success, 200);

        return apiResponse(0, 'Parcel could not be canceled.', '', 400);

    }

    public function show(ParcelRequst $request, $parcel_id)
    {
        $data['user_id'] = $request->user_login['login_id'];
        $parcel = $this->parcelService->getParcel($parcel_id); 

        if($parcel)
        return apiResponse(1, 'Parcel Detail.', $parcel, 200);      
        else
        return apiResponse(0, 'Valid Parcel id required.', [], 200);        
  
    }

    public function isParcelAccept()
    {
        $parcel = $this->parcelService->isRideAccept();

        if (!$parcel) {
            return apiResponse(1, 'Driver Not Found.', $parcel, 200);
        }

        return apiResponse(1, 'Accepted.', $parcel, 200); 
    }

    public function inProcessRide_OLD(Request $request)
    {
        $userId = $request->user_login['login_id'];

        $data = $this->parcelService->inProcessRide($userId);
        if ($data) {
            return apiResponse(1, 'In Process Ride.', $data, 200);
        }

        return apiResponse('1', 'Failed to get parcel.', '', 400);
    }
    
    public function inProcessRide(Request $request, $call_type=null)
    {                
        $userId    = $request->user_login['login_id'];
        $pendingRides = collect([ASSIGNED, ON_THE_WAY, DRIVER_ARRIVED, OTP_VERIFY, START_RIDE, PAYMENT_INITIATED, PAYMENT_COMPLETE, PENDING, QUEUED])
            ->map(function ($status) use ($userId) {        
                $parcel = Parcels::with(['user:id,name,mobile,photo', 'driver:id,name,mobile,model,brand,registration_number,latitude,longitude'])
            ->where('status', $status)
            ->where('user_id', $userId)
            ->orderBy('id', 'DESC')
            ->first(['id', 'user_id', 'driver_id', 'pickup_address', 'pickup_latitude', 'pickup_longitude','delivery_address','delivery_latitude', 'delivery_longitude', 'schedule_ride', 'distance', 'travel_time', 'final_price', 'status']);
                
            if ($parcel) {
            #$parcel->estimated_time = $status === COMPLETE ? null : "15 mins";
            $parcel->estimated_time = $parcel->travel_time . " mins";
            $parcel->distance_unit = "km";
            }
            return $parcel;
        })
        ->filter()
        ->first();

        if($call_type == 'inner'){
        return $pendingRides;
        }else{
        return apiResponse(1, '', $pendingRides, 200);
        }

    }

    public function rideStartStatus(ParcelRequst $request)
    {
        $userId = $request->user_login['login_id'];

        $data = $request->validated();

        $data = $this->parcelService->startRideStatus($userId, $data['parcel_id']);
        if ($data) {
            return apiResponse(1, 'In Process Ride.', $data, 200);
        }

        return apiResponse('0', 'There is not ride for this request.', '', 400);
    }

    public function rideTrack(Request $request)
    {
        $userId = $request->user_login['login_id'];
        $data = $this->parcelService->rideTrack($userId);
    }

    public function rideStatus(Request $request)
    {
        $userId = $request->user_login['login_id'];
        $parcelId = $request->input('parcel_id');
        $recentCompletedRide = Parcels::with('driver')
            ->where('user_id', $userId)
            ->where('id', $parcelId)
            ->first();
        $recentCompletedRide->status_name   = statusName($recentCompletedRide->status);   
        return apiResponse(1, 'Ride Status', $recentCompletedRide, 200);
    }

    public function recentRide(Request $request)
    {
        $userId = $request->user_login['login_id'];
        $recentCompletedRide = Parcels::with('driver.category')
            ->where('user_id', $userId)
            ->where('status', COMPLETE)
            ->orderBy('id', 'DESC')
            ->first();
        if($recentCompletedRide){
            $date = Carbon::parse($recentCompletedRide->created_at);
            $recentCompletedRide->currency          = currency_icon();
            $recentCompletedRide->created_at_mdfy   =  $date->format('j-M-Y H:i');
            $recentCompletedRide->category_photo_root_url = url('').'/category/';
             
            return apiResponse(1, 'Recent Ride', $recentCompletedRide, 200);
        }else{
            return apiResponse(0, 'There is no ride.', '', 200);
        }


    }

    public function allRides(Request $request)
    {
        $user_id = $request->user_login['login_id'];
        $trips = $this->parcelService->allRides($user_id);
        if(count($trips))
        return apiResponse(1, 'All Trips.', $trips, 200);
        else
        return apiResponse(0, 'No Trips.', $trips, 200);
    }
   
    public function couponList(Request $request)
    {
        $user_id = $request->user_login['login_id'];
        $couponArr = $this->parcelService->allCoupon();
        if(count($couponArr))
        return apiResponse(1, 'Coupon List.', $couponArr, 200);
        else
        return apiResponse(0, 'There is no active coupon.', [], 200);

    } 

    public function applyCoupon(ParcelRequst $request)
    {
        $data = $request->validated();
        $data['user_id'] = $request->user_login['login_id'];
        
        $coupon = Discount::find($data['coupon_id']);
        if($coupon && $data['original_price'])
        {
            if($data['original_price'] < $coupon->min_amount) {
                return apiResponse(0, 'This coupon requires a minimum amount of '. $coupon->min_amount . '.', [], 200);
            }
            $discountType       = $coupon->type;
            $discountValue      = $coupon->amount;
            $discountMinValue   = $coupon->min_amount;
            $originalPrice      = $data['original_price'];
            $finalPrice         = $originalPrice;
            $discount_precent_amount = 0;
            if ($discountType == 'flat') {
            $discount_precent_amount = $discountValue;
            $finalPrice -= $discountValue;
            } elseif ($discountType == 'percent') {
            $discount_precent_amount = ($originalPrice * $discountValue) / 100;
            $finalPrice -= $discount_precent_amount;
            }

            $finalPrice = max(0, $finalPrice);
         
            $api_data = [];
            $api_data['coupon_id']      = $coupon->id;
            $api_data['coupon_name']    = $coupon->name;
            $api_data['discount_type']  = $discountType;
            $api_data['discount_value'] = round($discount_precent_amount);
            $api_data['original_price'] = round($originalPrice);
            $api_data['estimated_price']= round($originalPrice);
            $api_data['final_price']    = round($finalPrice);
            $api_data['currency']       = currency_icon();
       
            return apiResponse(1, 'Coupon Apply Successfully.', $api_data, 200);

        } else {
            return apiResponse(0, 'There is no ride.', [], 200); 
        } 
    }

    public function cancelReason()
    {
        return apiResponse(1, 'Reason List.', reasonList(), 200);
    }

    public function ridePaymentSuccessOld(ParcelRequst $request)
    {
        $data = $request->validated();
        $data['user_id'] = $request->user_login['login_id'];

        $rideObj = Parcels::where('id', $data['parcel_id'])
            ->where('user_id', $data['user_id'])
            ->first(); 

        if($rideObj)
        {
            #$rideObj->status  = PAYMENT_COMPLETE;
            $rideObj->status  = COMPLETE;
            $rideObj->save();

            $this->parcelService->queuedToAssign($rideObj->driver_id);
 
            return apiResponse(1, 'Payment successfully.', $rideObj, 200); 
        }
        return apiResponse(0, 'There is no ride.', [], 200);
    }

    public function ridePaymentSuccess(ParcelRequst $request)
    {
    $data = $request->validated();
    $userId = $request->user_login['login_id'];

    if (empty($request->parcel_id)) {
        return apiResponse(0, 'Ride Id required.', [], 200);
    }

    $parcelId = $request->parcel_id;

    DB::beginTransaction();

    try {
        $parcel = Parcels::where('id', $parcelId)
                    ->where('user_id', $userId)
                    ->first();

        if (!$parcel) {
            DB::rollBack();
            return apiResponse(0, 'Ride not found for this user.', [], 200);
        }

        $parcel->status = COMPLETE;
        $parcel->save();

        DB::table('parcel_requests')
            ->where('parcel_id', $parcelId)
            ->where('driver_id', $parcel->driver_id)
            ->update(['status' => COMPLETE]);

        $this->parcelService->queuedToAssign($parcel->driver_id);

        DB::commit();
        return apiResponse(1, 'Ride Completed.', $parcel, 200);

    } catch (\Throwable $e) {
        DB::rollBack();
        return apiResponse(0, 'Failed to complete ride. Error: ' . $e->getMessage(), [], 500);
    }
    }
        

}
